/******************** SDbyIrDA.c ***********************/
/*-----------------------------------------------------*/

#include "XTWR-MCF51JE.h"
#include "SDbyIrDA.h"
#include "sci.h"
#include "IrDA.h"
#include "Fat.h"

/******************** USED VARIABLES *************************/

UINT8  u8sd_status;       /* Variable to indicate SD status */
UINT8  u8sd_command;      /* Variable to indicate commands for SD */
UINT8  u8irda_status;     /* Variable to indicate commands for IrDA */
UINT8  u8buffer[512];     /* Array to store information of the SD */
UINT16 u16buffer_index;   /* Variable to indicate position of buffer */
UINT16 u16file_counter;   /* Variable to indicate remaining bytes of a file */


/************************************************************************************/
/************************************************************************************/


void vfnMGC_init(void)
{
  /* 16MHz external clock source connected. */  
  MCGC2 = 0x36;                    // RANGE = 1; HGO = 1; ERCLKEN=1; EREFS = 1; BDIV = 000
  MCGC3 = 0x10;                    // DIV32 =1
  MCGC1 = 0xA0;                    // CLKS = 10; RDIV = 100; IREFS = 0
  while (MCGSC_IREFST){};          // Wait for Reference Status bit to update
  while (MCGSC_CLKST != 0b10){};   // Wait for clock status bits to update 
  MCGC3 = 0x5C;                    // PLLS =1; DIV32 = 1; VDIV = 1001 
  while (!MCGSC_PLLST){};          // Wait for PLL status bit to update
  while (!MCGSC_LOCK){};           // Wait for LOCK bit to set 
  MCGC1 = 0x20;                    // CLKS = 00; RDIV = 100; IREFS = 0
  while (MCGSC_CLKST != 0b11){};   // Wait for clock status bits to update 
}

/*----------------------------------------------------------------------------------*/

void vfnWelcome(void)
{
  /* Sends a message to indicate the application title */ 
  sendCR();
  vfnSCI_SendMsg(USED_PORT,"**********************************************************************"); 
  vfnSCI_SendChr(USED_PORT, CR);
  vfnSCI_SendChr(USED_PORT, LF);
  vfnSCI_SendMsg(USED_PORT,"Freescale Semiconductor. RTAC Americas"); 
  vfnSCI_SendChr(USED_PORT, CR);
  vfnSCI_SendChr(USED_PORT, LF);
  vfnSCI_SendMsg(USED_PORT,"IrDA driver and SD card file system on MM/JE Flexis families");
  vfnSCI_SendChr(USED_PORT, CR);
  vfnSCI_SendChr(USED_PORT, LF);
  vfnSCI_SendMsg(USED_PORT,"32-bit MCF51JE256 Device");
  sendCR(); 
}

/*----------------------------------------------------------------------------------*/

void vfnSDwelcome(void)
{
  /* Check the status of the SD card and sends messages */ 
  if (u8sd_status == NO_SD_CARD)
  {
    vfnSCI_SendMsg(USED_PORT,"SD card is not present!");  
    vfnSCI_SendChr(USED_PORT, CR);
    vfnSCI_SendChr(USED_PORT, LF);
    vfnSCI_SendMsg(USED_PORT,"System halted, press RESET button...");
    sendCR();
    DisableInterrupts;   //Disable interrupts
    for(;;){};
  }   
  else
  {
    vfnSCI_SendMsg(USED_PORT,"SD card detected...");  
    sendCR();
    
    //If the SD is ok, reads the Master Block of the memory
    FAT_Read_Master_Block();
    u8sd_status = FAT_FileOpen("FILE.TXT",MODIFY);
      
    if (u8sd_status == FILE_NOT_FOUND) 
    {
      //If the file does not exist, it is created
      u8sd_status = FAT_FileOpen("FILE.TXT",CREATE);  
      vfnSCI_SendMsg(USED_PORT,"FILE.TXT was created.");  
    } 
    else
    { 
      //If the exists, just a message is sent
      vfnSCI_SendMsg(USED_PORT,"FILE.TXT was found.");    
    }
        
    FAT_FileClose(); 
    sendCR();
    //These messages indicates functions of SW2 and SW4
    vfnSCI_SendMsg(USED_PORT,"Press Switch 2 once to write data onto FILE.TXT, press");
    vfnSCI_SendChr(USED_PORT, CR);
    vfnSCI_SendChr(USED_PORT, LF);
    vfnSCI_SendMsg(USED_PORT,"again to stop writting. Press Switch 4 to read the file.");   
    sendCR(); 
    
    //This part sets the initial values of these variables
    u16buffer_index = 0;
    u8sd_command = SD_INACTIVE;     
    u8irda_status = IrDA_INACTIVE;
  }
}

/*----------------------------------------------------------------------------------*/

void vfnReadFile(void)
{
  u8sd_command = SD_FILE_READING;
  //Indicates that the file will be read
  u8sd_status = FAT_FileOpen("FILE.TXT",READ);
  if (u8sd_status == FILE_FOUND) 
  {  
    //Sends an message to show the content of the file
    sendCR();
    vfnSCI_SendMsg(USED_PORT,"The content of FILE.TXT is:");     
    sendCR();
    //Indicates that data will go from SD to SCI through IrDA 
    u8irda_status = IrDA_SD_TO_SCI;
  
    do
    {  
      //Return of funtion FAT_FileRead indicates how many bytes were read
      //To read a complete file, this function should be invoked until it returns zero
      u16file_counter = FAT_FileRead(u8buffer); 
      for (u16buffer_index=0; u16buffer_index<u16file_counter; u16buffer_index++)
      {
        //Data from SD is sent modulated in IrDA
        vfnIrDA_transmission(u8buffer[u16buffer_index]);
      }
    }
    while (u16file_counter);
    
    //Sets these variables to inactive states
    u8irda_status = IrDA_INACTIVE;
    u8sd_command = SD_INACTIVE;
    u16buffer_index = 0;
    sendCR();    
  } 
  else
  {       
    //If the file was not found, the system is halted
    vfnSysHalt();
  }
  //Waits until button is released
  while( SW4_ON() ){};      
}

/*----------------------------------------------------------------------------------*/

void vfnWriteFile(void)
{
  if (u8sd_command == SD_INACTIVE)
  {
    //If is the first time that button is pressed, indicates that the file will be written
    u8sd_command = SD_FILE_WRITING;
    vfnSCI_SendMsg(USED_PORT,"Text will be saved on FILE.TXT");     
    sendCR();
    u16buffer_index = 0;
    //Indicates that data will go from SCI to SD
    u8irda_status = IrDA_SCI_TO_SD;
  }

  else if ( (u8sd_command == SD_FILE_WRITING) || (u16buffer_index >= 512) )
  {    
    //If is the second time that button is pressed, or the buffer reaches its maximum value
    //the buffer will be written on the file
    u8sd_status = FAT_FileOpen("FILE.TXT",MODIFY); 
    if (u8sd_status == FILE_FOUND) 
    {
      sendCR();
      if(u16buffer_index >= 512)
      { 
        //This message is sent when the buffer reaches the last position
        vfnSCI_SendMsg(USED_PORT,"Buffer size was reached.");
        vfnSCI_SendChr(USED_PORT, CR);
        vfnSCI_SendChr(USED_PORT, LF);
        u16buffer_index = 512;
      }
      
      //Writtes the u8buffer to the file
      FAT_FileWrite(u8buffer, u16buffer_index);
      FAT_FileClose(); 
      vfnSCI_SendMsg(USED_PORT,"Wrote text was saved on FILE.TXT");  
      sendCR(); 
      
      //Sets these variables to inactive states 
      u16buffer_index = 0;
      u8sd_command = SD_INACTIVE;
      u8irda_status = IrDA_INACTIVE;        
    }
    else
    { 
      //If the file was not found, the system is halted
      vfnSysHalt();
    }
  }
  //Waits until button is released
  while( SW2_ON() ){};            
    
}

/*----------------------------------------------------------------------------------*/

void vfnSysHalt(void)
{
  /* This funtion sends an error message and blocks the system */
  sendCR();
  vfnSCI_SendMsg(USED_PORT,"SD card was removed!");     
  vfnSCI_SendChr(USED_PORT, CR);                                    
  vfnSCI_SendChr(USED_PORT, LF);
  vfnSCI_SendMsg(USED_PORT,"System halted, press RESET button...");
  sendCR();
  DisableInterrupts;   // Disable interrupts 
  for(;;){};
}

/*----------------------------------------------------------------------------------*/